/************************************************************************
 * @file: Logging.h
 *
 * @version: 0.1
 *
 * @description: This header file contains definition and implementation for
 * Logging class. Logging is a utility class, used by all RTA components for
 * dumping the logs. Logging class will collect the logs and calls the
 * LoggingInterface API's for dumping.
 *
 * @authors: Jens Lorenz, jlorenz@de.adit-jv.com 2015
 *           Vijay Palaniswamy, vijay.palaniswamy@in.bosch.com 2015
 *
 * @copyright (c) 2015 Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/
#ifndef _ADIT_UTILITY_LOGGING_H_
#define _ADIT_UTILITY_LOGGING_H_

#include <sstream>
#include <string>
#include <algorithm>

#include "LoggingInterface.h"

namespace adit
{

namespace utility
{

class LoggingInterface;

class Logging
{
public:

    /**
     * Constructor Function
     * @param [IN] obj   Handle for LoggingInterface class used for displaying Logs
     * @param [IN] level Specify the Log level.
     */
    Logging(LoggingInterface& obj, const eLogLevel level) : mLoggingInterface(obj)
    {
        mLevel = std::min(LL_DEBUG, level);
        mIsLogActive = (mLevel <= mLoggingInterface.checkLogLevel());
    };
    virtual ~Logging(void)
    {
        send();
    };

    enum eEOL { endl };
    Logging& operator << (const eEOL)
    {
        send();
        return *this;
    };

    template<typename T>
    Logging& operator << (const T& data)
    {
        if (mIsLogActive)
        {
            mMessage << data;
        }
        return *this;
    };

private:

    inline void send(void)
    {
        void (LoggingInterface::*func[LL_MAX])(const std::string&) const =
        {
             &LoggingInterface::error,
             &LoggingInterface::warning,
             &LoggingInterface::info,
             &LoggingInterface::debug
        };

        if (mIsLogActive)
        {
            if (!mMessage.str().empty())
            {
                (mLoggingInterface.*(func[mLevel]))(mMessage.str());
                mMessage.str("");
                mMessage.clear();
            }
        }
    };

    std::ostringstream mMessage;
    eLogLevel          mLevel;
    bool               mIsLogActive;
    LoggingInterface&  mLoggingInterface;
};

} /* namespace utility */

} /* namespace adit */

#endif /* _ADIT_UTILITY_LOGGING_H_ */
